<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);
// session_start();
// require_once "../_db.php"; // Adjust path if needed
include "./include/head.php";

// ✅ Prevent double declaration of clean()
if (!function_exists('clean')) {
    function clean($data) {
        global $conn;
        return htmlspecialchars(strip_tags(trim(mysqli_real_escape_string($conn, $data))));
    }
}

// ✅ Authentication check
if (!isset($_SESSION['userid'])) {
    header("Location: login.php");
    exit();
}

$userid = $_SESSION['userid'];
$product_id = $_GET['product_id'] ?? '';

if (empty($product_id)) {
    echo "No product selected.";
    exit();
}

// ✅ Handle message or image upload
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $message = trim($_POST['message'] ?? '');
    $imagePath = null;

    // ✅ Upload image if provided
    if (isset($_FILES['image']) && $_FILES['image']['error'] === 0) {
        $ext = strtolower(pathinfo($_FILES['image']['name'], PATHINFO_EXTENSION));
        $allowed = ['jpg', 'jpeg', 'png', 'gif'];

        if (in_array($ext, $allowed)) {
            $uploadDir = "uploads/";
            if (!is_dir($uploadDir)) {
                mkdir($uploadDir, 0777, true);
            }

            $newFilename = uniqid("chat_") . "." . $ext;
            $targetPath = $uploadDir . $newFilename;

            if (move_uploaded_file($_FILES['image']['tmp_name'], $targetPath)) {
                $imagePath = $newFilename; // Save just the filename
            }
        }
    }

    // ✅ Save message if text or image is present
    if (!empty($message) || $imagePath) {
        $stmt = $conn->prepare("INSERT INTO product_messages (product_id, sender_role, sender_id, receiver_id, message, image, sent_at, seen) VALUES (?, 'seller', ?, 'admin', ?, ?, NOW(), 0)");
        $stmt->bind_param("ssss", $product_id, $userid, $message, $imagePath);
        $stmt->execute();
        $stmt->close();

        header("Location: chat.php?product_id=" . urlencode($product_id));
        exit();
    }
}

// ✅ Mark admin messages as seen
$updateSeen = $conn->prepare("UPDATE product_messages SET seen = 1 WHERE product_id = ? AND sender_role = 'admin' AND receiver_id = ?");
$updateSeen->bind_param("ss", $product_id, $userid);
$updateSeen->execute();
$updateSeen->close();

// ✅ Fetch messages
$mstmt = $conn->prepare("SELECT * FROM product_messages WHERE product_id = ? AND (sender_id = ? OR receiver_id = ?) ORDER BY sent_at ASC");
$mstmt->bind_param("sss", $product_id, $userid, $userid);
$mstmt->execute();
$messages = $mstmt->get_result()->fetch_all(MYSQLI_ASSOC);
$mstmt->close();

// ✅ Get product name
$product_name = "Product";
$pstmt = $conn->prepare("SELECT product_name FROM products WHERE product_id = ?");
$pstmt->bind_param("s", $product_id);
$pstmt->execute();
$pstmt->bind_result($product_name);
$pstmt->fetch();
$pstmt->close();
?>

<?php
include "./include/navbar.php";

include "sidebar.php";
?>

<!-- ✅ WhatsApp-style chat UI -->
<style>
    .chat-wrapper {
        background: #e5ddd5;
        font-family: 'Segoe UI', sans-serif;
    }

    .chat-box {
        max-height: 65vh;
        overflow-y: auto;
        padding: 20px;
        background: #f0f0f0;
        border-radius: 10px;
    }

    .message-bubble {
        padding: 10px 15px;
        border-radius: 15px;
        margin-bottom: 12px;
        max-width: 75%;
        word-wrap: break-word;
    }

    .admin-bubble {
        background: #fff;
        text-align: left;
        align-self: flex-start;
        border-top-left-radius: 0;
    }

    .user-bubble {
        background: #dcf8c6;
        text-align: right;
        align-self: flex-end;
        border-top-right-radius: 0;
        margin-left: auto;
    }

    .message-wrapper {
        display: flex;
        flex-direction: column;
    }

    .image-msg {
        max-width: 220px;
        margin-top: 5px;
        border-radius: 8px;
    }

    .chat-form {
        display: flex;
        flex-direction: column;
        gap: 10px;
        margin-top: 15px;
    }

    .chat-input-wrapper {
        display: flex;
        align-items: center;
        background: white;
        border-radius: 30px;
        padding: 8px 12px;
        box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
    }

    .chat-input-wrapper textarea {
        border: none;
        flex: 1;
        resize: none;
        padding: 8px 12px;
        border-radius: 20px;
        font-size: 15px;
        height: 38px;
        overflow-y: auto;
        outline: none;
    }

    .attachment-btn {
        display: flex;
        align-items: center;
        justify-content: center;
        color: #888;
        margin-right: 10px;
        cursor: pointer;
        font-size: 18px;
        position: relative;
    }

    .attachment-btn input[type="file"] {
        display: none;
    }

    .send-btn {
        background-color: #128C7E;
        color: white;
        border: none;
        border-radius: 50%;
        padding: 10px;
        font-size: 16px;
        cursor: pointer;
        margin-left: 8px;
        display: flex;
        align-items: center;
        justify-content: center;
    }

    .send-btn:hover {
        background-color: #075E54;
    }
</style>

<div class="container chat-wrapper mx-auto px-4 py-6">
    <h2 class="text-lg font-bold mb-4">Chat about: <?php echo htmlspecialchars($product_name); ?></h2>

    <div class="chat-box">
        <?php if (empty($messages)) : ?>
            <p class="text-gray-600">No messages yet.</p>
        <?php else : ?>
            <?php foreach ($messages as $msg) : ?>
                <div class="message-wrapper">
                    <div class="message-bubble <?php echo $msg['sender_role'] === 'admin' ? 'admin-bubble' : 'user-bubble'; ?>">
                        <?php if (!empty($msg['message'])) : ?>
                            <p><?php echo nl2br(htmlspecialchars($msg['message'])); ?></p>
                        <?php endif; ?>

                        <?php if (!empty($msg['image'])) : ?>
                            <img src="uploads/<?php echo htmlspecialchars($msg['image']); ?>" alt="image" class="image-msg">
                        <?php endif; ?>

                        <div style="font-size: 0.75rem; color: gray; margin-top: 5px;">
                            <?php echo date('M d, Y h:i A', strtotime($msg['sent_at'])); ?>
                        </div>
                    </div>
                </div>
            <?php endforeach; ?>
        <?php endif; ?>
    </div>

    <form class="chat-form" method="POST" enctype="multipart/form-data">
        <div class="chat-input-wrapper">
            <label for="image-upload" class="attachment-btn" title="Attach Image">
                <i class="fas fa-paperclip"></i>
                <input type="file" name="image" id="image-upload" accept="image/*">
            </label>

            <textarea name="message" rows="1" placeholder="Type a message..."></textarea>

            <button type="submit" class="send-btn">
                <i class="fas fa-paper-plane"></i>
            </button>
        </div>
    </form>

    <!-- FontAwesome for icons -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" />
</div>

<script>
    // ✅ Scroll to bottom
    window.onload = function () {
        const chatBox = document.querySelector('.chat-box');
        chatBox.scrollTop = chatBox.scrollHeight;
    };
</script>
